#!/bin/bash
# Copyright ETSI 2019-2021
# See: https://forge.etsi.org/etsi-forge-copyright-statement.txt

LOGFILE=openapi-validation.log
DOCKER_IMAGE=etsiforge/swagger-cli:4.0.3

echo -e "\n------ Switching to $1 folder --------"
cd $1

echo "------ Checking for previous logs ------"
if [ -f "$LOGFILE" ] ; then
    echo "Found previous log. Removing it"
    rm "$LOGFILE"
fi

if [ -x "$(command -v docker)" ] ; then
    RUN_IN_DOCKER=0
    echo "Using dockerized validator ($DOCKER_IMAGE)"
    docker pull "$DOCKER_IMAGE"
	#docker pull -q "$DOCKER_IMAGE" --> Pull quietly only from Ubuntu 20.04
else
    RUN_IN_DOCKER=1
    if [ ! -x "$(command -v swagger-cli)" ] ; then
        echo "Validator swagger-cli not found. Quitting."
        exit 1
    else
        echo "Using local validator ($(which swagger-cli))"
    fi
fi

function validate {
    if [ $RUN_IN_DOCKER == 0 ] ; then
	   bname=$(basename "$1")
       docker run --rm -v $(dirname $(realpath $1)):"/specs" $DOCKER_IMAGE swagger-cli validate "/specs/$bname"
    else
        swagger-cli validate "$1"
    fi
}

echo -e "\n------ Validating all YAML files (may takes several minutes) ------"
# If there are no YAML file, simply exit
ls | grep -q yaml
found_yaml=$?
if [ ! $found_yaml ] ; then 
    echo "-- No YAML files."
    exit 0
fi

fres=0
LOG=""
ERR=""
for i in *.yaml ;  do
    echo -e "----  Validating $i ...  "    
    MSG=$(validate "$i" 2>&1)
	echo "$MSG"
    res=$?
    if [ ! $res == 0 ] ; then 
        ERR="$ERR$MSG\n"
    fi 
    LOG="$LOG$MSG\n"
    fres=$(($fres||$res))
done

echo -e "\n-- Final validator returns $fres." | tee -a $LOGFILE
if [ $fres == 0 ] ; then
   echo "No errors found, all files validate the OpenAPI definition. Below the complete log." | tee -a $LOGFILE
else
   echo "Some errors found, see below the details." | tee -a $LOGFILE
fi

if [ ! "$ERR" == "" ] ; then 
    echo -e "\n---- Errors detected ----" | tee -a $LOGFILE
    echo -e $ERR  | tee -a $LOGFILE
fi

echo -e "\n---- Complete log of validation ----" | tee -a $LOGFILE
echo -e $LOG | tee -a $LOGFILE

echo -e "\n------ Content of the folder ------" | tee -a $LOGFILE
ls | tee -a $LOGFILE

# Exit code needed for jenkins to know the verdict of the build

exit $fres
