import logging
import json
from pathlib import Path
from subprocess import run
from shutil import which

from pycrate_asn1c.asnproc import *

def reconstrainInteger (filename):
    Path('temp.asn').write_text(Path(filename).read_text().replace("18446744073709551615", "65536"))
    return 'temp.asn'

filesWithBigInts = [
    '102232-1/LI-PS-PDU.asn',
    '102232-3/IPAccessPDU.asn',
    '102232-4/L2AccessPDU.asn'
]

asn1c_path = ""
change_path_to_unix = False

def fix_path(path):
    if change_path_to_unix:
        return "./" + path.replace("\\","/")
    else:
        return path

def syntaxCheckASN (fileList):
    """
    Performs ASN syntax checking on a list of filenames (or pathlib Paths)

    :param fileList: List of filenames (str or Pathlib Path)
    :returns: Dict with result, return code and message for each filename

    Calls the open-source asn1c compiler with the "syntax only" option.
    As a result, asn1c must be available to run.
    """
    results = {}
    for file in fileList:
        try:
            if file.as_posix() in filesWithBigInts:
                newFile = reconstrainInteger(str(file))
                p = run([asn1c_path, '-E', fix_path(newFile)], capture_output=True)
                Path(newFile).unlink()
            else:
                p = run([asn1c_path, '-E', fix_path(str(file))], capture_output=True)
            if (p.returncode != 0):
                errorMessage = p.stderr.decode().splitlines()[0]
                if errorMessage.startswith('   Value "18446744073709551615" at line'):
                    results[str(file)] = { 'ok' : True}
                    continue
                results[str(file)] = {
                    'ok'   : False,
                    'code' : p.returncode,
                    'message'  : p.stderr.decode().splitlines()[0]
                }
            else:
                results[str(file)] = {
                    'ok'   : True
                }            
        except Exception as ex:
            raise ex
            results[str(file)] = {
                'ok'   : False,
                'code' : -1,
                'message'  : f"{ex!r}"
            }
    return results


duplicateObjects = {
    '102232-1/LI-PS-PDU.asn' : [
        'CCPayload',
        'IRIPayload',
        'Location'
    ],
    'testing/mod1.asn' : [
        'ClashField'
    ],
    'ASN/TS32298_GPRSChargingDataTypes.asn' : [
        'GSNAddress'
    ],
}
def fixDuplicateObjects(filename):
    stringContent = filename.read_text()
    for object in duplicateObjects[filename.as_posix()]:
        stringContent = stringContent.replace(f'{object} ::=', f'Native{object} ::=')
        stringContent = stringContent.replace(f'SEQUENCE OF {object}', f'SEQUENCE OF Native{object}')
        #stringContent = sub(f"]\\w{object}", f"] Native{object}", stringContent)

    Path('temp.asn').write_text(stringContent)
    return 'temp.asn'


def compileAllTargets (compileTargets):
    """
    Attempts to compile a set of compile targets using the pycrate ASN1 tools

    :param compileTargets: list of compile targets, each of which is a list of filenames
    :returns: A dict of outcome against the first filename of each compile target. Return code and message are included for failures.

    For each compile target (list of filenames) the first filename is assumed
    to be the "primary" file. This doesn't have any relavance to the compilation,
    but will be used as the identifier when reporting any compile errors.
    The compilation is performed by the pycrate ASN compile functions; errors
    are caught as exceptions and rendered into a list. 
    
    Unfortunately, the pycrate compiler doesn't report line numbers.
    The asn1c compiler does, but doesn't properly handle identifiers with the 
    same name in different modules; as this occurs multiple times in TS 33.108,
    we can't use it.
    """
    results = {}
    for target in compileTargets:
        firstTarget = target[0]
        logging.debug(f"Compiling {firstTarget}")
        try:
            fileTexts = []
            fileNames = []
            GLOBAL.clear()
            for filename in target:
                pFile = Path(filename)
                if pFile.as_posix() in duplicateObjects:
                    tmpFile = Path(fixDuplicateObjects(pFile))
                    fileTexts.append(tmpFile.read_text())
                    #tmpFile.unlink()
                else:
                    fileTexts.append(pFile.read_text())
                fileNames.append(filename)
                logging.debug (f"  Loading {filename}")
            compile_text(fileTexts, filenames = fileNames)
            results[str(firstTarget)] = {
                'ok' : True,
            }
        except Exception as ex:
            results[str(firstTarget)] = {
                'ok'   : False,
                'code' : -1,
                'message'  : f"{ex!r}"
            }
            continue
    return results



def processResults (results, stageName):
    """
    Counts the number of errors and writes out the output per filename

    :param results: List of filenames (str or Pathlib Path)
    :param stageName: Name to decorate the output with
    :returns: The number of files which had errors
    """    
    print("")
    errorCount = sum([1 for r in results.values() if not r['ok']])
    logging.info(f"{errorCount} {stageName} errors encountered")
    
    print(f"{'-':-<60}")
    print(f"{stageName} results:")
    print(f"{'-':-<60}")
    for filename, result in results.items():
        print(f" {filename:.<55}{'..OK' if result['ok'] else 'FAIL'}")
        if not result['ok']:
            if isinstance(result['message'], list):
                for thing in result['message']:
                    print(f"    {thing['message']}")
            else:
                print(f"    {result['message']}")
    
    print(f"{'-':-<60}")
    print(f"{stageName} errors: {errorCount}")
    print(f"{'-':-<60}")
 
    return errorCount


if __name__ == '__main__':
    logging.info ('Searching for ASN1C')
    asn1c_path = which("asn1c")
    if asn1c_path is None:
        raise Exception ("No asn1c executable found. Please install asn1c")
    logging.info (f"asn1c found at {asn1c_path}")
    if asn1c_path.lower().endswith("bat"):
        logging.info (f"asn1c is a batch file, so assume path separators need to be changed")
        change_path_to_unix = True
    

    logging.info('Searching for ASN.1 files')
    fileList = list(Path(".").rglob("*.asn1")) + list(Path(".").rglob("*.asn"))
    logging.info(f'{len(fileList)} ASN.1 files found')
    for file in fileList:
        logging.debug(f'  {file}')
    
    #ignoreList = Path('testing/asn_ignore.txt').read_text().splitlines()
    ignoreList = []
    ignoredFiles = []
    for ignore in ignoreList:
        logging.debug(f'Ignoring pattern {ignore}')
        for file in fileList:
            if ignore in str(file):
                ignoredFiles.append(file)
                logging.debug(f" Ignoring {str(file)} as contains {ignore}")
    ignoredFiles = list(set(ignoredFiles))
    logging.info(f'{len(ignoredFiles)} files ignored')
    for file in ignoredFiles:
        logging.debug(f'  {file}')
    
    fileList = [file for file in fileList if file not in ignoredFiles]
    logging.info(f'{len(fileList)} files to process')
    for file in fileList:
        logging.debug(f'  {file}')

    if len(fileList) == 0:
        logging.warning ("No files specified")
        exit(0)
    
    logging.info("Parsing ASN1 files")
    parseResults = syntaxCheckASN(fileList)
    if processResults(parseResults, "Parsing") > 0:
        exit(-1)

    logging.info ("Getting compile targets")
    compileTargets = json.loads(Path('asn_compile_targets.json').read_text())
    logging.info (f"{len(compileTargets)} compile targets found")

    compileResults = compileAllTargets(compileTargets)
    if processResults(compileResults, "Compiling") > 0:
        exit(-1)
    
    exit(0)
