#
#	delete_old_branches.sh
#
#	Script to delete older than a year branches 
#
#	(c) 2025 by Miguel Angel Reina Ortega
#	License: BSD 3-Clause License. See the LICENSE file for further details.
#
#!/bin/bash

#Parameters

#${CI_API_V4_URL} -> 1
echo "CI_API_V4_URL:" $1
#${CI_PROJECT_ID} -> 2
echo "CI_PROJECT_ID:" $2

#!/bin/bash

PRIVATE_TOKEN=$3
ONE_YEAR_AGO=$(date -d '1 year ago' --iso-8601=seconds)
DRY_RUN=$4 # Set to false to actually delete
LOG_FILE="deleted_branches_$(date +%Y%m%d_%H%M%S).log"

# ==== Helper: URL encode ====
urlencode() {
  jq -nr --arg v "$1" '$v|@uri'
}

# ==== Initialize ====
page=1
deleted_branches=0
cd logs

echo "Dry-run mode: $DRY_RUN"
echo "Log file: $LOG_FILE"
echo "" > "$LOG_FILE" # Start fresh

# ==== Loop over paginated branch list ====
while true; do
  response=$(curl --silent --header "PRIVATE-TOKEN: $PRIVATE_TOKEN" \
    "$1/projects/$2/repository/branches?page=$page")

  count=$(echo "$response" | jq length)
  if [ "$count" -eq 0 ]; then
    break
  fi

  jq -c '.[]' <<< "$response" > /tmp/branches.json
  while read -r branch; do
    branch_name=$(printf '%s' "$branch" | jq -r '.name')
    is_protected=$(printf '%s' "$branch" | jq -r '.protected')
    last_commit_date=$(printf '%s' "$branch" | jq -r '.commit.committed_date')

    if [ "$is_protected" = "true" ]; then
      echo "Skipping protected branch: $branch_name"
      continue
    fi

    if [[ "$last_commit_date" < "$ONE_YEAR_AGO" ]]; then
      ((deleted_branches++))
      echo "==> OLD branch: $branch_name (last commit: $last_commit_date)" | tee -a "$LOG_FILE"
      if [ "$DRY_RUN" = "true" ]; then
        echo "[DRY-RUN] Would delete: $branch_name (last commit: $last_commit_date)" | tee -a "$LOG_FILE"
      else
        encoded_branch=$(urlencode "$branch_name")
        delete_response=$(curl --silent --request DELETE \
          --header "PRIVATE-TOKEN: $PRIVATE_TOKEN" \
          "$1/projects/$PROJECT_ID/repository/branches/$encoded_branch")

        echo "Deleted: $branch_name" | tee -a "$LOG_FILE"
      fi
    else
      echo "Keeping branch: $branch_name (last commit: $last_commit_date)" | tee -a "$LOG_FILE"
    fi
  done < /tmp/branches.json

  ((page++))
done

echo "Total deleted branches: $deleted_branches" | tee -a "$LOG_FILE"
if [ "$DRY_RUN" = "true" ]; then
  echo "[DRY-RUN] No branches actually deleted." | tee -a "$LOG_FILE"
fi

echo "Done. Deleted branches logged to: $LOG_FILE"

