#!/bin/bash
# Copyright ETSI 2019-2021
# See: https://forge.etsi.org/etsi-forge-copyright-statement.txt

LOGFILE=openapi-validation.txt
DOCKER_IMAGE=etsiforge/swagger-cli:4.0.3

echo -e "\n------ Switching to $1 folder --------"
cd $1
ls

echo "------ Checking for previous logs ------"
if [ -f "$LOGFILE" ] ; then
    echo "Found previous log. Removing it"
    rm "$LOGFILE"
fi

if [ -x "$(command -v docker)" ] ; then
    RUN_IN_DOCKER=0
    echo "Using dockerized validator ($DOCKER_IMAGE)"
    docker pull "$DOCKER_IMAGE"
	#docker pull -q "$DOCKER_IMAGE" --> Pull quietly only from Ubuntu 20.04
else
    RUN_IN_DOCKER=1
    if [ ! -x "$(command -v swagger-cli)" ] ; then
        echo "Validator swagger-cli not found. Quitting."
        exit 1
    else
        echo "Using local validator ($(which swagger-cli))"
    fi
fi

function validate {
    echo -e "----  Validating $i:  "    
    if [ $RUN_IN_DOCKER == 0 ] ; then
	   bname=$(basename "$1")
       docker run --rm -v $(dirname $(realpath $1)):"/specs" $DOCKER_IMAGE swagger-cli validate "/specs/$bname"
    else
        swagger-cli validate "$1"
    fi
}

echo -e "\n------ Validating all YAML files (may takes several minutes) ------"
# If there are no YAML file, simply exit
ls | grep -q yaml
found_yaml=$?
if [ $found_yaml == 1 ] ; then 
    echo "-- No YAML files." >> $LOGFILE
    exit 0
fi

#Check if externals directory exists and contains yaml files
if [ -d "externals" ] && [ -n "$(find externals -name '*.yaml')" ]; then
    echo "Processing external YAML files..."
    for external_file in externals/*.yaml; do
        base_name=$(basename "$external_file")
        if [ -f "./$base_name" ]; then
            # Get last commit timestamps for both files
            external_timestamp=$(cd externals && git log -1 --format=%ct "$(basename $external_file)")
            current_timestamp=$(git log -1 --format=%ct "$base_name" 2>/dev/null || echo "0")
            
            if [ "$external_timestamp" -gt "$current_timestamp" ]; then
                echo "Updating $base_name from externals (newer version found in Git history)"
                cp "$external_file" "./$base_name"
            else
                echo "Keeping existing $base_name (newer or same age in Git history)"
            fi
        else
            # File doesn't exist in current directory, copy it
            echo "Copying new file $base_name from externals"
            cp "$external_file" "./"
        fi
    done
else
    echo "No external YAML files to process"
fi

#Proceed with validation
fres=0
LOG=""
ERR=""
for i in ./*.yaml ;  do

    MSG=$(validate "$i" 2>&1)
    res=$?
    if [ ! $res == 0 ] ; then 
        ERR="$ERR$MSG\n"
    fi 
    LOG="$LOG$MSG\n"
	echo -e "----  Validating $i ... valid"
    fres=$(($fres||$res))
done

echo -e "\n-- Final validator returns $fres." | tee -a $LOGFILE
if [ $fres == 0 ] ; then
   echo "No errors found, all files validate the OpenAPI definition" | tee -a $LOGFILE
else
   echo "Some errors found by the OpenAPI validation" | tee -a $LOGFILE
fi

if [ ! "$ERR" == "" ] ; then 
    echo -e "\n---- Errors detected ----" >> $LOGFILE
    echo -e $ERR >> $LOGFILE
fi

echo -e "\n---- Complete log of validation ----" >> $LOGFILE
echo -e $LOG >> $LOGFILE

echo -e "\n------ Content of the folder ------" >> $LOGFILE
ls >> $LOGFILE

if [ ! -s $LOGFILE ] ;then
	rm "$LOGFILE"
else
	echo "To see the full output of the OpenAPI validation, download the archived artifact named '$LOGFILE'."
fi

# Exit code needed for jenkins to know the verdict of the build

exit $fres
